-- Databricks notebook source
-- MAGIC %md
-- MAGIC
-- MAGIC ### # -- SETUP: WIDGETS AND CATALOG/SCHEMA CREATION
-- MAGIC ### # 
-- MAGIC ### # -- We will use the same catalog but separate schemas for each layer.
-- MAGIC

-- COMMAND ----------

CREATE WIDGET TEXT catalog DEFAULT "default";
CREATE WIDGET TEXT schema DEFAULT "supply_chain";

-- COMMAND ----------

-- MAGIC %md
-- MAGIC ### -- Set the current catalog
-- MAGIC ### USE CATALOG IDENTIFIER(:catalog);

-- COMMAND ----------

CREATE CATALOG IF NOT EXISTS IDENTIFIER(:catalog);
USE CATALOG IDENTIFIER(:catalog);

CREATE SCHEMA IF NOT EXISTS IDENTIFIER(:schema);
USE SCHEMA IDENTIFIER(:schema);

-- COMMAND ----------

-- MAGIC %md
-- MAGIC <img src="/Workspace/Shared/intelligent_supply_chain/supplier_scorecard/Images/medalion Architecture.png" width="400">
-- MAGIC
-- MAGIC  **🪙 Bronze → Raw landing zone** <br>
-- MAGIC      
-- MAGIC  **🔧 Silver → Clean, validated, normalized data**<br>
-- MAGIC      
-- MAGIC **🏆 Gold  → Analytical, enriched datasets for dashboards**

-- COMMAND ----------

-- MAGIC %md
-- MAGIC
-- MAGIC **BRONZE LAYER (RAW DATA)**
-- MAGIC
-- MAGIC **This layer simulates raw, uncleaned ingestion. Data is loaded as-is.**
-- MAGIC
-- MAGIC **Data Characteristics:**
-- MAGIC - Records directly inserted (e.g., through INSERT INTO commands)<br>
-- MAGIC - May contain messy, inconsistent, or duplicate values
-- MAGIC
-- MAGIC **📂 Bronze Raw Tables**
-- MAGIC - 🛠️ `vehicle_master`
-- MAGIC - 🧾 `supplier_master`
-- MAGIC - 🔩 `part_master`
-- MAGIC - 🚚 `supplier_sourced`
-- MAGIC - 📊 `warranty_claims`
-- MAGIC
-- MAGIC <img src="/Workspace/Shared/intelligent_supply_chain/supplier_scorecard/Images/bronze1.png" width="150">
-- MAGIC
-- MAGIC

-- COMMAND ----------

CREATE OR REPLACE TABLE vehicle_master (
    vehicle_program STRING NOT NULL PRIMARY KEY,
    vehicle_model_year INT,
    vehicle_model STRING,
    vehicle_price DOUBLE
);

-- COMMAND ----------

CREATE OR REPLACE TABLE supplier_master (
    supplier_number INT NOT NULL PRIMARY KEY,
    supplier_name STRING,
    supplier_address STRING,
    supplier_city STRING,
    supplier_postalcode STRING,
    supplier_region STRING,
    supplier_country STRING,
    supplier_duns_number INT,
    supplier_email STRING,
    supplier_phone STRING,
    latitude DOUBLE,
    longitude DOUBLE
);


-- COMMAND ----------

CREATE OR REPLACE TABLE part_master (
    part_number INT NOT NULL PRIMARY KEY,
    part_name STRING,
    unspc_code STRING,
    part_category STRING,
    part_weight DOUBLE,
    part_material STRING,
    part_lead_time_days INT,
    part_cost DOUBLE
);

-- COMMAND ----------

CREATE OR REPLACE TABLE supplier_sourced (
  supplier_sourced_id STRING NOT NULL PRIMARY KEY,
  part_number INT NOT NULL,
  supplier_number INT NOT NULL,
  quantity_sourced INT,
  date_requested DATE,
  date_received DATE,
  date_shipped DATE,
  expected_date_received DATE,
  expected_date_shipped DATE,
  part_price DOUBLE,
  freight_cost DOUBLE,
  FOREIGN KEY (supplier_number) REFERENCES supplier_master(supplier_number),
  FOREIGN KEY (part_number) REFERENCES part_master(part_number)
);

-- COMMAND ----------

CREATE OR REPLACE TABLE warranty_claims (
  warranty_claims_id STRING NOT NULL PRIMARY KEY,
  vehicle_program STRING NOT NULL,
  part_number INT,
  date_filed DATE,
  claim_desc STRING,
  claim_cost DOUBLE,
  labor_cost DOUBLE,
  part_cost DOUBLE,
  FOREIGN KEY (vehicle_program) REFERENCES vehicle_master(vehicle_program),
  FOREIGN KEY (part_number) REFERENCES part_master(part_number)
);

-- COMMAND ----------

-- MAGIC %md
-- MAGIC **🧪 DATA INSERTION (Sample Data Load)**
-- MAGIC
-- MAGIC Mock data is inserted into the **Bronze layer tables** to simulate a realistic supply chain environment and enable development, testing, and dashboarding without external data sources.
-- MAGIC
-- MAGIC **📥 Populated Tables**
-- MAGIC - 🚗 `vehicle_master`
-- MAGIC - 🏭 `supplier_master`
-- MAGIC - 🔧 `part_master`
-- MAGIC - 📦 `supplier_sourced`
-- MAGIC - 🛠️ `warranty_claims`
-- MAGIC
-- MAGIC **🎯 Why Insert Mock Data?**
-- MAGIC - Allows full pipeline execution (Bronze ➜ Silver ➜ Gold)
-- MAGIC - Enables KPI + dashboard development before real data exists
-- MAGIC - Supports validation of schemas, joins, and transformations
-- MAGIC
-- MAGIC <img src="/Workspace/Shared/intelligent_supply_chain/supplier_scorecard/Images/DataInsert.png" width="200">

-- COMMAND ----------

INSERT INTO part_master (
    part_number,
    part_name,
    unspc_code,
    part_category,
    part_weight,
    part_material,
    part_lead_time_days,
    part_cost
)
VALUES
  (20000, 'Air Filter', '25101501', 'Engine', 0.5, 'Synthetic Fiber', 7, 10.99),
  (20001, 'Oil Filter', '25101502', 'Engine', 0.4, 'Steel', 5, 9.99),
  (20002, 'Spark Plug', '25102001', 'Ignition', 0.1, 'Iridium', 3, 12.99),
  (20003, 'Brake Pad', '25102501', 'Braking', 1.2, 'Ceramic', 10, 14.99),
  (20004, 'Exhaust System', '25103001', 'Exhaust', 8.0, 'Stainless Steel', 14, 19.99),
  (20005, 'Catalytic Converter', '25103002', 'Exhaust', 6.5, 'Platinum Alloy', 21, 24.99),
  (20006, 'Fuel Injector', '25103501', 'Fuel', 0.3, 'Aluminum', 12, 29.99),
  (20007, 'Fuel Pump', '25103502', 'Fuel', 1.8, 'Steel', 15, 34.99),
  (20008, 'Alternator', '25104001', 'Electrical', 5.0, 'Aluminum', 18, 39.99),
  (20009, 'Starter Motor', '25104002', 'Electrical', 4.5, 'Steel', 16, 44.99),
  (20010, 'Battery', '25104501', 'Electrical', 18.0, 'Lithium-ion', 20, 49.99),
  (20011, 'Tire', '30101001', 'Wheels', 9.0, 'Rubber', 8, 114.99),
  (20012, 'Wheel Bearing', '30101501', 'Wheels', 0.7, 'Steel', 6, 59.99),
  (20013, 'Shock Absorber', '30102001', 'Suspension', 2.5, 'Steel', 13, 64.99),
  (20014, 'Steering Wheel', '30102501', 'Steering', 1.6, 'Leather', 9, 69.99),
  (20015, 'Radiator', '25105001', 'Cooling', 7.2, 'Aluminum', 11, 79.99),
  (20016, 'Water Pump', '25105002', 'Cooling', 2.1, 'Steel', 10, 49.99),
  (20017, 'Timing Belt', '25105501', 'Engine', 0.8, 'Rubber', 7, 29.99),
  (20018, 'Drive Shaft', '30103001', 'Transmission', 6.0, 'Steel', 15, 89.99),
  (20019, 'Clutch Plate', '30103501', 'Transmission', 1.3, 'Ceramic', 8, 39.99),
  (20020, 'Headlight Assembly', '25106001', 'Lighting', 1.5, 'Plastic', 5, 54.99),
  (20021, 'Tail Light Assembly', '25106002', 'Lighting', 1.2, 'Plastic', 5, 44.99),
  (20022, 'Windshield Wiper', '25106501', 'Body', 0.3, 'Rubber', 3, 14.99),
  (20023, 'Window Regulator', '25106502', 'Body', 1.1, 'Steel', 7, 34.99),
  (20024, 'Door Handle', '25107001', 'Body', 0.2, 'Plastic', 4, 9.99),
  (20025, 'Rearview Mirror', '25107002', 'Body', 0.6, 'Glass', 6, 24.99),
  (20026, 'Seat Belt', '25107501', 'Safety', 0.7, 'Nylon', 5, 19.99),
  (20027, 'Airbag Module', '25107502', 'Safety', 1.0, 'Fabric', 12, 99.99),
  (20028, 'Horn', '25108001', 'Electrical', 0.4, 'Steel', 3, 12.99),
  (20029, 'Fuse Box', '25108002', 'Electrical', 0.9, 'Plastic', 7, 29.99),
  (20030, 'Engine Mount', '25108501', 'Engine', 2.0, 'Rubber', 9, 39.99),
  (20031, 'Transmission Mount', '25108502', 'Transmission', 2.2, 'Rubber', 10, 44.99),
  (20032, 'Control Arm', '30104001', 'Suspension', 3.5, 'Steel', 13, 59.99),
  (20033, 'Ball Joint', '30104002', 'Suspension', 0.5, 'Steel', 8, 19.99),
  (20034, 'CV Joint', '30104501', 'Transmission', 1.4, 'Steel', 11, 34.99);

-- COMMAND ----------

INSERT INTO vehicle_master (vehicle_program, vehicle_model_year, vehicle_model, vehicle_price)
VALUES
    ('2021-Fjellstrom', 2021, 'Fjellstrom XP', 35000.00),
    ('2022-Fjellstrom', 2022, 'Fjellstrom XP', 38000.00),
    ('2023-Fjellstrom', 2023, 'Fjellstrom XP', 37500.00),
    ('2024-Fjellstrom', 2024, 'Fjellstrom XP', 37300.00),
    ('2025-Fjellstrom', 2025, 'Fjellstrom XP', 37000.00),
    ('2021-Nordixa', 2021, 'Nordixa', 17850.00),
    ('2022-Nordixa', 2022, 'Nordixa', 17850.00),
    ('2023-Nordixa', 2023, 'Nordixa', 19850.00),
    ('2024-Nordixa', 2024, 'Nordixa', 19500.00),
    ('2025-Nordixa', 2025, 'Nordixa', 19500.00),
    ('2022-Volkara', 2022, 'Volkara', 52100.00),
    ('2023-Volkara', 2023, 'Volkara', 52100.00),
    ('2024-Volkara', 2024, 'Volkara', 52500.00),
    ('2025-Volkara', 2025, 'Volkara', 52900.00),
    ('2024-Lysvik', 2024, 'Lysvik', 115000.00),
    ('2025-Lysvik', 2025, 'Lysvik', 117000.00),
    ('2025-Kaelori', 2025, 'Kaelori', 35700.00);

-- COMMAND ----------

INSERT INTO supplier_master (
  supplier_number, supplier_name, supplier_address, supplier_city, 
  supplier_postalcode, supplier_region, supplier_country, supplier_duns_number, 
  supplier_email, supplier_phone, latitude, longitude
)
VALUES
  (10000,'TurboStar Inc.','1234 Velocity Drive','Tokyo','162-0000','Kanto','Japan',123456789,'turbostarinc10000@email.com','(201) 384-9271',35.6895,139.6917),
  (10001,'Veloce Motors','456 Speedway Boulevard','Sydney','2000','New South Wales','Australia',987654321,'velocemotors10001@email.com','(312) 495-1832',-33.8688,151.2093),
  (10002,'Gearhive Industries','789 Turbo Avenue','Bengaluru','560100','Karnataka','India',555555555,'gearhiveindustries10002@email.com','(415) 672-3498',12.9716,77.5946),
  (10003,'Pitstop Performance','321 Racing Drive','Detroit','48265','Michigan','United States',333333333,'pitstopperformance10003@email.com','(518) 203-7645',42.3314,-83.0458),
  (10004,'Eurotech Automotive','901 Precision Road','London','W1D 1BS','England','United Kingdom',777777777,'eurotechautomotive10004@email.com','(626) 481-2390',51.5074,-0.1278),
  (10005,'Dyno Dynamics','234 Dragon Road','Shanghai','200001','Shanghai','China',666666666,'dynodynamics10005@email.com','(734) 592-1843',31.2304,121.4737),
  (10006,'Brazilian Bolt Works','567 Samba Street','Sao Paulo','01311-000','Sao Paulo','Brazil',888888888,'brazilianboltworks10006@email.com','(845) 317-9204',-23.5505,-46.6333),
  (10007,'RPM Racing Supplies','890 Fast Lane','Paris','75008','Ile-de-France','France',444444444,'rpmracingsupplies10007@email.com','(956) 204-3851',48.8566,2.3522),
  (10008,'Koreana Auto Parts','345 Seoul Speedway','Seoul','06164','Seoul','South Korea',999999999,'koreanaautoparts10008@email.com','(307) 498-6723',37.5665,126.9780),
  (10009,'Autocraft Canada','678 Motorway Drive','Toronto','M5B 1B8','Ontario','Canada',222222222,'autocraftcanada10009@email.com','(419) 583-2047',43.6532,-79.3832),
  (10010,'TurboStar Inc.','1234 Velocity Drive','Tokyo','162-0000','Kanto','Japan',123456789,'turbostarinc10010@email.com','(520) 384-9272',35.6895,139.6917),
  (10011,'Veloce Motors','456 Speedway Boulevard','Sydney','2000','New South Wales','Australia',987654321,'velocemotors10011@email.com','(621) 495-1833',-33.8688,151.2093),
  (10012,'Gearhive Industries','789 Turbo Avenue','Bengaluru','560100','Karnataka','India',555555555,'gearhiveindustries10012@email.com','(415) 672-3499',12.9716,77.5946),
  (10013,'Pitstop Performance','321 Racing Drive','Detroit','48265','Michigan','United States',333333333,'pitstopperformance10013@email.com','(518) 203-7646',42.3314,-83.0458),
  (10014,'Eurotech Automotive','901 Precision Road','London','W1D 1BS','England','United Kingdom',777777777,'eurotechautomotive10014@email.com','(626) 481-2391',51.5074,-0.1278),
  (10015,'Dyno Dynamics','234 Dragon Road','Shanghai','200001','Shanghai','China',666666666,'dynodynamics10015@email.com','(734) 592-1844',31.2304,121.4737),
  (10016,'Brazilian Bolt Works','567 Samba Street','Sao Paulo','01311-000','Sao Paulo','Brazil',888888888,'brazilianboltworks10016@email.com','(845) 317-9205',-23.5505,-46.6333),
  (10017,'RPM Racing Supplies','890 Fast Lane','Paris','75008','Ile-de-France','France',444444444,'rpmracingsupplies10017@email.com','(956) 204-3852',48.8566,2.3522),
  (10018,'Koreana Auto Parts','345 Seoul Speedway','Seoul','06164','Seoul','South Korea',999999999,'koreanaautoparts10018@email.com','(307) 498-6724',37.5665,126.9780),
  (10019,'Autocraft Canada','678 Motorway Drive','Toronto','M5B 1B8','Ontario','Canada',222222222,'autocraftcanada10019@email.com','(419) 583-2048',43.6532,-79.3832),
  (10020,'Alpine Engines','12 Mountain Rd','Munich','80331','Bavaria','Germany',111111111,'alpineengines10020@email.com','(530) 384-9273',48.1351,11.5820);


-- COMMAND ----------

SELECT COUNT(*) FROM isupply.supply_chain.supplier_master;


-- COMMAND ----------

INSERT INTO 
  supplier_sourced (supplier_sourced_id, part_number, supplier_number, quantity_sourced, date_requested, date_received, date_shipped, expected_date_received, expected_date_shipped, part_price, freight_cost)
VALUES
  ('SS0001', 20000, 10000, 500, '2025-01-02', '2025-01-15', '2025-01-05', '2025-01-17', '2025-01-07',10.99, 50.00),
  ('SS0002', 20001, 10001, 750, '2025-01-05', '2025-01-20', '2025-01-07','2025-01-25' ,'2025-01-08' ,9.99, 75.00),
  ('SS0003', 20002, 10002, 200, '2025-01-10', '2025-01-25', '2025-01-11', '2025-01-24', '2025-01-11',12.99, 30.00),
  ('SS0004', 20003, 10003, 1000, '2025-01-12', '2025-02-01', '2025-01-13', '2025-02-05','2025-01-11' ,14.99, 100.00),
  ('SS0005', 20004, 10004, 250, '2025-01-15', '2025-02-05', '2025-01-17', '2025-02-01', '2025-02-01',19.99, 50.00),
  ('SS0006', 20005, 10005, 300, '2025-01-18', '2025-02-10', '2025-01-20', '2025-02-11', '2025-01-23',24.99, 60.00),
  ('SS0007', 20006, 10006, 400, '2025-01-20', '2025-02-15', '2025-01-20', '2025-02-17', '2025-01-19',29.99, 80.00),
  ('SS0008', 20007, 10007, 150, '2025-01-22', '2025-02-20', '2025-01-26', '2025-02-21', '2025-02-01',34.99, 40.00),
  ('SS0009', 20008, 10008, 600, '2025-01-25', '2025-03-01', '2025-01-26', '2025-02-23','2025-01-27' ,39.99, 90.00),
  ('SS0010', 20009, 10009, 800, '2025-01-28', '2025-03-05', '2025-01-28', '2025-03-07', '2025-01-26',44.99, 120.00),
  ('SS0011', 20010, 10000, 350, '2025-02-01', '2025-02-15', '2025-02-05', '2025-02-18','2025-02-18' ,49.99, 70.00),
  ('SS0012', 20011, 10001, 900, '2025-02-05', '2025-02-20', '2025-02-08', '2025-02-20', '2025-02-10',114.99, 110.00),
  ('SS0013', 20012, 10002, 450, '2025-02-10', '2025-02-25', '2025-02-11', '2025-02-22', '2025-02-11',59.99, 85.00),
  ('SS0014', 20013, 10003, 550, '2025-02-12', '2025-03-01', '2025-02-12', '2025-03-01', '2025-02-12' ,64.99, 95.00),
  ('SS0015', 20014, 10004, 650, '2025-02-15', '2025-03-05', '2025-02-19', '2025-03-05', '2025-02-27' ,69.99, 105.00),
  ('SS0001', 20000, 10000, 500, '2025-01-02', '2025-01-15', '2025-01-05', '2025-01-17', '2025-01-07',10.99, 50.00),
  ('SS0002', 20001, 10001, 750, '2025-01-05', '2025-01-20', '2025-01-07','2025-01-25' ,'2025-01-08' ,9.99, 75.00),
  ('SS0003', 20002, 10002, 200, '2025-01-10', '2025-01-25', '2025-01-11', '2025-01-24', '2025-01-11',12.99, 30.00),
  ('SS0004', 20003, 10003, 1000, '2025-01-14', '2025-02-05', '2025-01-13', '2025-02-05','2025-01-11' ,14.99, 100.00),
  ('SS0005', 20004, 10004, 250, '2025-01-15', '2025-02-05', '2025-01-17', '2025-02-01', '2025-02-01',19.99, 50.00),
  ('SS0006', 20005, 10005, 300, '2025-01-20', '2025-02-10', '2025-01-20', '2025-02-11', '2025-01-23',24.99, 60.00),
  ('SS0007', 20006, 10006, 400, '2025-01-16', '2025-02-15', '2025-01-20', '2025-02-17', '2025-01-11',29.99, 80.00),
  ('SS0008', 20007, 10007, 150, '2025-01-23', '2025-02-20', '2025-01-25', '2025-02-11', '2025-02-09',34.99, 40.00),
  ('SS0009', 20008, 10008, 600, '2025-01-23', '2025-03-01', '2025-01-26', '2025-02-21','2025-01-01' ,39.99, 90.00),
  ('SS0010', 20009, 10009, 800, '2025-01-26', '2025-03-05', '2025-01-27', '2025-03-09', '2025-01-25',44.99, 120.00),
  ('SS0011', 20010, 10000, 350, '2025-02-03', '2025-02-15', '2025-02-05', '2025-02-19','2025-02-18' ,49.99, 70.00),
  ('SS0012', 20011, 10001, 900, '2025-02-01', '2025-02-20', '2025-02-08', '2025-02-26', '2025-02-10',114.99, 110.00),
  ('SS0013', 20012, 10002, 450, '2025-02-01', '2025-02-25', '2025-02-15', '2025-02-27', '2025-02-17',59.99, 85.00),
  ('SS0014', 20013, 10003, 550, '2025-02-05', '2025-03-01', '2025-02-13', '2025-03-05', '2025-02-12' ,64.99, 95.00),
  ('SS0015', 20014, 10004, 650, '2025-02-06', '2025-03-05', '2025-02-17', '2025-03-05', '2025-02-09' ,69.99, 105.00),
  ('SS0101',20000,10000,500,'2025-03-01','2025-03-15','2025-03-05','2025-03-13','2025-03-03',10.99,50.00),
  ('SS0102',20001,10001,750,'2025-03-02','2025-03-18','2025-03-07','2025-03-25','2025-03-08',9.99,75.00),
  ('SS0103',20002,10002,200,'2025-03-03','2025-03-20','2025-03-11','2025-03-24','2025-03-11',12.99,30.00),
  ('SS0104',20003,10003,1000,'2025-03-04','2025-03-22','2025-03-13','2025-03-28','2025-03-11',14.99,100.00),
  ('SS0105',20004,10004,250,'2025-03-05','2025-03-25','2025-03-17','2025-03-29','2025-03-21',19.99,50.00),
  ('SS0106',20005,10005,300,'2025-03-06','2025-03-27','2025-03-20','2025-03-31','2025-03-23',24.99,60.00),
  ('SS0107',20006,10006,400,'2025-03-07','2025-03-28','2025-03-21','2025-04-01','2025-03-19',29.99,80.00),
  ('SS0108',20007,10007,150,'2025-03-08','2025-03-29','2025-03-25','2025-04-02','2025-03-21',34.99,40.00),
  ('SS0109',20008,10008,600,'2025-03-09','2025-04-01','2025-03-26','2025-04-03','2025-03-27',39.99,90.00),
  ('SS0110',20009,10009,800,'2025-03-10','2025-04-02','2025-03-28','2025-04-05','2025-03-26',44.99,120.00),
  ('SS0111',20010,10010,350,'2025-03-11','2025-04-04','2025-03-29','2025-04-06','2025-03-28',49.99,70.00),
  ('SS0112',20011,10011,900,'2025-03-12','2025-04-05','2025-04-01','2025-04-07','2025-04-02',114.99,110.00),
  ('SS0113',20012,10012,450,'2025-03-13','2025-04-06','2025-04-02','2025-04-08','2025-04-03',59.99,85.00),
  ('SS0114',20013,10013,550,'2025-03-14','2025-04-07','2025-04-03','2025-04-09','2025-04-04',64.99,95.00),
  ('SS0115',20014,10014,650,'2025-03-15','2025-04-08','2025-04-05','2025-04-07','2025-04-04',69.99,105.00),
  ('SS0116',20000,10015,500,'2025-03-16','2025-04-09','2025-04-06','2025-04-11','2025-04-07',11.99,55.00),
  ('SS0117',20001,10016,750,'2025-03-17','2025-04-10','2025-04-07','2025-04-12','2025-04-08',10.49,65.00),
  ('SS0118',20002,10017,200,'2025-03-18','2025-04-11','2025-04-08','2025-04-13','2025-04-09',13.49,35.00),
  ('SS0119',20003,10018,1000,'2025-03-19','2025-04-12','2025-04-09','2025-04-14','2025-04-10',15.49,105.00),
  ('SS0120',20004,10019,250,'2025-03-20','2025-04-13','2025-04-10','2025-04-15','2025-04-11',20.49,55.00),
  ('SS0121',20005,10020,300,'2025-03-20','2025-04-14','2025-04-11','2025-04-16','2025-04-12',25.49,65.00),
  ('SS0122',20006,10011,400,'2025-03-22','2025-04-15','2025-04-12','2025-04-17','2025-04-11',30.49,85.00),
  ('SS0123',20007,10012,150,'2025-03-23','2025-04-16','2025-04-13','2025-04-18','2025-04-14',35.49,45.00),
  ('SS0124',20008,10013,600,'2025-03-24','2025-04-17','2025-04-14','2025-04-19','2025-04-15',40.49,95.00),
  ('SS0125',20009,10014,800,'2025-03-25','2025-04-18','2025-04-15','2025-04-20','2025-04-16',45.49,125.00),
  ('SS0126',20010,10000,350,'2025-03-26','2025-04-19','2025-04-16','2025-04-21','2025-04-17',50.49,75.00),
  ('SS0127',20011,10001,900,'2025-03-22','2025-04-20','2025-04-17','2025-04-22','2025-04-18',115.49,115.00),
  ('SS0128',20012,10002,450,'2025-03-28','2025-04-21','2025-04-18','2025-04-23','2025-04-19',60.49,90.00),
  ('SS0129',20013,10003,550,'2025-03-29','2025-04-22','2025-04-19','2025-04-24','2025-04-20',65.49,100.00),
  ('SS0130',20014,10004,650,'2025-03-30','2025-04-23','2025-04-20','2025-04-25','2025-04-21',70.49,110.00),
  ('SS0131',20000,10005,500,'2025-03-31','2025-04-24','2025-04-21','2025-04-26','2025-04-22',12.49,60.00),
  ('SS0132',20001,10006,750,'2025-04-01','2025-04-25','2025-04-22','2025-04-27','2025-04-23',11.49,70.00),
  ('SS0133',20002,10007,200,'2025-04-02','2025-04-26','2025-04-23','2025-04-28','2025-04-24',14.49,40.00),
  ('SS0134',20003,10008,1000,'2025-04-02','2025-04-27','2025-04-24','2025-04-29','2025-04-25',16.49,110.00),
  ('SS0135',20004,10009,250,'2025-04-02','2025-04-28','2025-04-25','2025-04-30','2025-04-26',21.49,60.00),
  ('SS0136',20005,10010,300,'2025-04-02','2025-04-29','2025-04-26','2025-05-01','2025-04-27',26.49,70.00),
  ('SS0137',20006,10011,400,'2025-04-06','2025-04-30','2025-04-27','2025-05-02','2025-04-28',31.49,90.00),
  ('SS0138',20007,10012,150,'2025-04-06','2025-05-01','2025-04-28','2025-05-03','2025-04-29',36.49,50.00),
  ('SS0139',20008,10013,600,'2025-04-08','2025-05-02','2025-04-29','2025-05-04','2025-04-30',41.49,100.00),
  ('SS0140',20009,10014,800,'2025-04-09','2025-05-03','2025-04-30','2025-05-05','2025-05-01',46.49,130.00),
  ('SS0141',20010,10015,350,'2025-04-11','2025-05-04','2025-05-01','2025-05-06','2025-05-02',51.49,80.00),
  ('SS0142',20011,10016,900,'2025-04-11','2025-05-05','2025-05-02','2025-05-07','2025-05-03',116.49,120.00),
  ('SS0143',20012,10017,450,'2025-04-12','2025-05-06','2025-05-03','2025-05-08','2025-05-04',61.49,95.00),
  ('SS0144',20013,10018,550,'2025-04-13','2025-05-07','2025-05-04','2025-05-09','2025-05-05',66.49,105.00),
  ('SS0145',20014,10019,650,'2025-04-14','2025-05-08','2025-05-05','2025-05-10','2025-05-06',71.49,115.00),
  ('SS0146',20000,10020,500,'2025-04-15','2025-05-09','2025-05-06','2025-05-11','2025-05-07',12.99,55.00),
  ('SS0147',20001,10011,750,'2025-04-16','2025-05-10','2025-05-07','2025-05-12','2025-05-08',10.99,65.00),
  ('SS0148',20002,10012,200,'2025-04-17','2025-05-11','2025-05-08','2025-05-13','2025-05-09',13.99,35.00),
  ('SS0149',20003,10013,1000,'2025-04-18','2025-05-12','2025-05-09','2025-05-14','2025-05-10',15.99,105.00),
  ('SS0150',20004,10014,250,'2025-04-19','2025-05-13','2025-05-10','2025-05-15','2025-05-11',20.99,60.00),
  ('SS0151',20005,10000,300,'2025-04-20','2025-05-14','2025-05-11','2025-05-16','2025-05-12',25.99,70.00),
  ('SS0152',20006,10001,400,'2025-04-21','2025-05-15','2025-05-12','2025-05-17','2025-05-13',30.99,90.00),
  ('SS0153',20007,10002,150,'2025-04-22','2025-05-16','2025-05-13','2025-05-18','2025-05-14',35.99,50.00),
  ('SS0154',20008,10003,600,'2025-04-23','2025-05-17','2025-05-14','2025-05-19','2025-05-15',40.99,100.00),
  ('SS0155',20009,10004,800,'2025-04-24','2025-05-18','2025-05-15','2025-05-20','2025-05-16',45.99,130.00),
  ('SS0156',20010,10005,350,'2025-04-25','2025-05-19','2025-05-16','2025-05-21','2025-05-17',50.99,80.00),
  ('SS0157',20011,10006,900,'2025-04-26','2025-05-20','2025-05-17','2025-05-22','2025-05-18',115.99,120.00),
  ('SS0158',20012,10007,450,'2025-04-27','2025-05-21','2025-05-18','2025-05-23','2025-05-19',60.99,95.00),
  ('SS0159',20013,10008,550,'2025-04-28','2025-05-22','2025-05-19','2025-05-24','2025-05-20',65.99,105.00),
  ('SS0160',20014,10009,650,'2025-04-29','2025-05-23','2025-05-20','2025-05-25','2025-05-21',70.99,115.00),
  ('SS0161',20000,10010,500,'2025-04-30','2025-05-24','2025-05-21','2025-05-26','2025-05-22',13.49,56.00),
  ('SS0162',20001,10011,750,'2025-05-01','2025-05-25','2025-05-22','2025-05-27','2025-05-23',11.49,66.00),
  ('SS0163',20002,10012,200,'2025-05-02','2025-05-26','2025-05-23','2025-05-28','2025-05-24',14.49,36.00),
  ('SS0164',20003,10013,1000,'2025-05-03','2025-05-27','2025-05-24','2025-05-29','2025-05-25',16.49,106.00),
  ('SS0165',20004,10014,250,'2025-05-03','2025-05-28','2025-05-25','2025-05-30','2025-05-26',21.49,61.00),
  ('SS0166',20005,10015,300,'2025-05-03','2025-05-29','2025-05-26','2025-05-28','2025-05-27',26.49,71.00),
  ('SS0167',20006,10016,400,'2025-05-06','2025-05-30','2025-05-27','2025-06-01','2025-05-28',31.49,91.00),
  ('SS0168',20007,10017,150,'2025-05-07','2025-05-31','2025-05-28','2025-06-02','2025-05-29',36.49,51.00),
  ('SS0169',20008,10018,600,'2025-05-08','2025-06-01','2025-05-29','2025-06-03','2025-05-30',41.49,101.00),
  ('SS0170',20009,10019,800,'2025-05-09','2025-06-02','2025-05-30','2025-06-04','2025-05-31',46.49,131.00),


  
  ('SS0151',20005,10000,300,'2025-04-20','2025-05-14','2025-05-11','2025-05-12','2025-05-10',25.99,70.00),
  ('SS0152',20006,10001,400,'2025-04-21','2025-05-15','2025-05-12','2025-05-14','2025-05-11',30.99,90.00),
  ('SS0153',20007,10002,150,'2025-04-22','2025-05-16','2025-05-13','2025-05-11','2025-05-10',35.99,50.00),
  ('SS0154',20008,10003,600,'2025-04-23','2025-05-17','2025-05-14','2025-05-12','2025-05-12',40.99,100.00),
  ('SS0155',20009,10004,800,'2025-04-24','2025-05-18','2025-05-15','2025-07-13','2025-05-12',45.99,130.00),
  ('SS0156',20010,10005,350,'2025-04-25','2025-05-19','2025-05-16','2025-07-16','2025-05-17',50.99,80.00),
  ('SS0157',20011,10006,900,'2025-04-26','2025-05-20','2025-05-17','2025-07-16','2025-05-18',115.99,120.00),
  ('SS0158',20012,10007,450,'2025-04-27','2025-05-21','2025-05-18','2025-07-16','2025-05-19',60.99,95.00),
  ('SS0159',20013,10008,550,'2025-04-28','2025-05-22','2025-05-19','2025-07-16','2025-05-20',65.99,105.00),
  ('SS0160',20014,10009,650,'2025-04-29','2025-05-23','2025-05-20','2025-07-16','2025-05-21',70.99,115.00),
  ('SS0161',20000,10010,500,'2025-04-30','2025-05-24','2025-05-21','2025-05-26','2025-05-22',13.49,56.00),
  ('SS0162',20001,10011,750,'2025-05-01','2025-05-25','2025-05-22','2025-07-27','2025-05-23',11.49,66.00),
  ('SS0163',20002,10012,200,'2025-05-02','2025-05-26','2025-05-23','2025-07-16','2025-05-24',14.49,36.00),
  ('SS0164',20003,10013,1000,'2025-05-03','2025-05-27','2025-05-24','2025-05-16','2025-05-25',16.49,106.00),
  ('SS0165',20004,10014,250,'2025-05-03','2025-05-28','2025-05-25','2025-05-16','2025-05-26',21.49,61.00),
  ('SS0166',20005,10015,300,'2025-05-03','2025-05-29','2025-05-26','2025-05-16','2025-05-27',26.49,71.00),
  ('SS0167',20006,10016,400,'2025-05-06','2025-05-30','2025-05-27','2025-05-01','2025-05-28',31.49,91.00),
  ('SS0168',20007,10017,150,'2025-05-07','2025-05-31','2025-05-28','2025-05-02','2025-05-29',36.49,51.00),
  ('SS0169',20008,10018,600,'2025-05-08','2025-06-01','2025-05-29','2025-05-03','2025-05-30',41.49,101.00),
  ('SS0170',20009,10019,800,'2025-05-09','2025-06-02','2025-05-30','2025-05-04','2025-05-31',46.49,131.00);




-- COMMAND ----------

INSERT INTO warranty_claims (warranty_claims_id, vehicle_program, part_number, date_filed, claim_desc, claim_cost, labor_cost, part_cost)
VALUES
  -- Original claims
  ('WC0001', '2025-Fjellstrom', 20011, '2024-12-05', 'Tire failure due to manufacturing defect', 200.00, 50.00, 150.00),
  ('WC0002', '2025-Nordixa', 20011, '2024-12-10', 'Tire blowout due to faulty valve stem', 250.00, 75.00, 175.00),
  ('WC0003', '2025-Volkara', 20010, '2024-12-15', 'Battery failure due to internal short circuit', 300.00, 100.00, 200.00),
  ('WC0004', '2024-Lysvik', 20009, '2024-12-20', 'Starter motor failure due to faulty solenoid', 400.00, 150.00, 250.00),
  ('WC0005', '2025-Kaelori', 20007, '2024-12-25', 'Fuel pump failure due to clogged fuel filter', 500.00, 200.00, 300.00),
  ('WC0006', '2025-Fjellstrom', 20011, '2025-01-01', 'Tire failure due to uneven tire wear caused by faulty alignment', 220.00, 60.00, 160.00),
  ('WC0007', '2025-Nordixa', 20010, '2025-01-05', 'Battery failure due to faulty charging system', 280.00, 90.00, 190.00),
  ('WC0008', '2025-Volkara', 20007, '2025-01-10', 'Fuel pump failure due to faulty fuel pressure regulator', 550.00, 220.00, 330.00),
  ('WC0009', '2025-Fjellstrom', 20011, '2025-01-15', 'Tire failure due to manufacturing defect', 240.00, 70.00, 170.00),
  ('WC0010', '2025-Nordixa', 20009, '2025-01-20', 'Starter motor failure due to faulty starter relay', 380.00, 140.00, 240.00),
  ('WC0011', '2025-Volkara', 20007, '2025-01-25', 'Fuel pump failure due to clogged fuel injector', 520.00, 210.00, 310.00),
  ('WC0012', '2024-Lysvik', 20010, '2025-02-01', 'Battery failure due to internal short circuit', 320.00, 110.00, 210.00),
  ('WC0013', '2025-Kaelori', 20011, '2025-02-05', 'Tire failure due to faulty tire sensor', 260.00, 80.00, 180.00),
  ('WC0014', '2025-Nordixa', 20009, '2025-02-10', 'Starter motor failure due to faulty starter solenoid', 420.00, 160.00, 260.00),
  ('WC0015', '2025-Fjellstrom', 20007, '2025-02-15', 'Fuel pump failure due to faulty fuel pump relay', 580.00, 230.00, 350.00),
  ('WC0016', '2025-Volkara', 20010, '2025-02-20', 'Battery failure due to faulty battery management system', 340.00, 120.00, 220.00),
  ('WC0017', '2025-Fjellstrom', 20011, '2025-02-22', 'Tire failure due to manufacturing defect', 280.00, 90.00, 190.00),
  ('WC0018', '2025-Nordixa', 20009, '2025-02-25', 'Starter motor failure due to faulty starter motor bearings', 440.00, 170.00, 270.00),
  ('WC0019', '2025-Volkara', 20007, '2025-02-28', 'Fuel pump failure due to clogged fuel filter', 600.00, 240.00, 360.00),
  ('WC0020', '2024-Lysvik', 20011, '2025-03-01', 'Tire failure due to faulty tire valve stem', 300.00, 100.00, 200.00),
  ('WC0021', '2025-Kaelori', 20010, '2025-03-02', 'Battery failure due to internal short circuit', 380.00, 140.00, 240.00),
  ('WC0022', '2025-Fjellstrom', 20009, '2025-03-03', 'Starter motor failure due to faulty starter relay', 420.00, 160.00, 260.00),
  ('WC0023', '2025-Nordixa', 20007, '2025-03-04', 'Fuel pump failure due to faulty fuel pressure regulator', 580.00, 230.00, 350.00),
  ('WC0024', '2025-Volkara', 20011, '2025-01-12', 'Tire failure due to manufacturing defect', 280.00, 90.00, 190.00),
  ('WC0025', '2025-Fjellstrom', 20010, '2025-01-18', 'Battery failure due to faulty charging system', 320.00, 110.00, 210.00),
  ('WC0026', '2025-Nordixa', 20009, '2025-01-22', 'Starter motor failure due to faulty starter solenoid', 440.00, 170.00, 270.00),
  ('WC0027', '2025-Volkara', 20007, '2025-01-28', 'Fuel pump failure due to clogged fuel injector', 600.00, 240.00, 360.00),
  ('WC0028', '2024-Lysvik', 20011, '2025-02-02', 'Tire failure due to faulty tire sensor', 300.00, 100.00, 200.00),
  ('WC0029', '2025-Kaelori', 20010, '2025-02-08', 'Battery failure due to internal short circuit', 380.00, 140.00, 240.00),
  ('WC0030', '2025-Fjellstrom', 20009, '2025-02-12', 'Starter motor failure due to faulty starter motor bearings', 420.00, 160.00, 260.00),
  ('WC0031', '2025-Nordixa', 20007, '2025-02-18', 'Fuel pump failure due to faulty fuel pump relay', 580.00, 230.00, 350.00),
  ('WC0032', '2025-Volkara', 20011, '2025-02-20', 'Tire failure due to manufacturing defect', 280.00, 90.00, 190.00),
  ('WC0033', '2025-Fjellstrom', 20010, '2025-02-22', 'Battery failure due to faulty battery management system', 320.00, 110.00, 210.00),
  ('WC0034', '2025-Nordixa', 20009, '2025-02-25', 'Starter motor failure due to faulty starter solenoid', 440.00, 170.00, 270.00),
  ('WC0035', '2025-Volkara', 20007, '2025-02-28', 'Fuel pump failure due to clogged fuel filter', 600.00, 240.00, 360.00),
  ('WC0036', '2024-Lysvik', 20011, '2025-01-02', 'Tire failure due to faulty tire valve stem', 300.00, 100.00, 200.00),
  ('WC0037', '2025-Kaelori', 20010, '2025-01-08', 'Battery failure due to internal short circuit', 380.00, 140.00, 240.00),
  ('WC0038', '2025-Fjellstrom', 20009, '2025-01-12', 'Starter motor failure due to faulty starter relay', 420.00, 160.00, 260.00),
  ('WC0039', '2025-Nordixa', 20007, '2025-01-18', 'Fuel pump failure due to faulty fuel pressure regulator', 580.00, 230.00, 350.00),
  ('WC0040', '2025-Volkara', 20011, '2025-01-20', 'Tire failure due to manufacturing defect', 280.00, 90.00, 190.00),
  ('WC0041', '2025-Fjellstrom', 20010, '2025-01-22', 'Battery failure due to faulty charging system', 320.00, 110.00, 210.00),
  ('WC0042', '2025-Nordixa', 20009, '2025-01-25', 'Starter motor failure due to faulty starter solenoid', 440.00, 170.00, 270.00),
  ('WC0043', '2025-Volkara', 20007, '2025-01-28', 'Fuel pump failure due to clogged fuel injector', 600.00, 240.00, 360.00),
  ('WC0044', '2024-Lysvik', 20011, '2025-02-01', 'Tire failure due to faulty tire sensor', 300.00, 100.00, 200.00),
  ('WC0045', '2025-Kaelori', 20010, '2025-02-05', 'Battery failure due to internal short circuit', 380.00, 140.00, 240.00),
  ('WC0046', '2025-Fjellstrom', 20009, '2025-02-10', 'Starter motor failure due to faulty starter motor bearings', 420.00, 160.00, 260.00),
  ('WC0047', '2025-Nordixa', 20007, '2025-02-15', 'Fuel pump failure due to faulty fuel pump relay', 580.00, 230.00, 350.00),
  ('WC0048', '2025-Volkara', 20011, '2025-02-18', 'Tire failure due to manufacturing defect', 280.00, 90.00, 190.00),
  ('WC0049', '2025-Fjellstrom', 20010, '2025-02-20', 'Battery failure due to faulty battery management system', 320.00, 110.00, 210.00),
  ('WC0050', '2025-Nordixa', 20009, '2025-02-22', 'Starter motor failure due to faulty starter solenoid', 440.00, 170.00, 270.00),
  ('WC0051', '2025-Kaelori', 20008, '2025-03-05', 'Alternator failure due to faulty regulator', 420.00, 150.00, 270.00),
  ('WC0052', '2025-Volkara', 20012, '2025-03-07', 'Wheel bearing failure due to lubrication defect', 310.00, 120.00, 190.00),
  ('WC0053', '2025-Fjellstrom', 20013, '2025-03-09', 'Shock absorber leakage caused by poor seal quality', 280.00, 100.00, 180.00),
  ('WC0054', '2025-Nordixa', 20014, '2025-03-11', 'Steering wheel defect due to material crack', 350.00, 130.00, 220.00),
  ('WC0055', '2024-Lysvik', 20006, '2025-03-13', 'Fuel injector clogging due to residue', 400.00, 140.00, 260.00),
  ('WC0056', '2025-Kaelori', 20002, '2025-03-15', 'Spark plug misfire from manufacturing defect', 220.00, 70.00, 150.00),
  ('WC0057', '2025-Volkara', 20005, '2025-03-17', 'Catalytic converter failure due to overheating', 600.00, 200.00, 400.00),
  ('WC0058', '2025-Nordixa', 20004, '2025-03-19', 'Exhaust system corrosion issue', 500.00, 180.00, 320.00),
  ('WC0059', '2025-Fjellstrom', 20001, '2025-03-21', 'Oil filter leak from defective seal', 260.00, 80.00, 180.00),
  ('WC0060', '2024-Lysvik', 20000, '2025-03-23', 'Air filter tearing caused by poor assembly', 210.00, 60.00, 150.00),
  ('WC0061', '2025-Kaelori', 20009, '2025-03-25', 'Starter motor overheating due to wiring fault', 430.00, 160.00, 270.00),
  ('WC0062', '2025-Nordixa', 20003, '2025-03-27', 'Brake pad premature wear from defective material', 310.00, 110.00, 200.00),
  ('WC0063', '2025-Volkara', 20000, '2025-03-29', 'Air filter collapse under pressure', 240.00, 80.00, 160.00),
  ('WC0064', '2025-Fjellstrom', 20014, '2025-03-30', 'Steering wheel misalignment due to shaft defect', 360.00, 140.00, 220.00),
  ('WC0065', '2024-Lysvik', 20008, '2025-04-01', 'Alternator diode failure', 410.00, 150.00, 260.00),
  ('WC0066', '2025-Kaelori', 20013, '2025-04-02', 'Shock absorber mount crack', 300.00, 100.00, 200.00),
  ('WC0067', '2025-Volkara', 20011, '2025-04-03', 'Tire sidewall defect causing blowout', 280.00, 90.00, 190.00),
  ('WC0068', '2025-Nordixa', 20005, '2025-04-04', 'Catalytic converter inefficiency', 520.00, 190.00, 330.00),
  ('WC0069', '2025-Fjellstrom', 20006, '2025-04-05', 'Fuel injector leakage at seal', 390.00, 120.00, 270.00),
  ('WC0070', '2024-Lysvik', 20012, '2025-04-06', 'Wheel bearing noise due to defective steel', 340.00, 130.00, 210.00),

  -- New unique claims (different descriptions and costs)
  ('WC0071', '2025-Fjellstrom', 20011, '2025-04-10', 'Tire tread separation at high speed', 310.00, 120.00, 190.00),
  ('WC0072', '2025-Nordixa', 20011, '2025-04-12', 'Tire sidewall bulge from impact damage', 270.00, 90.00, 180.00),
  ('WC0073', '2025-Volkara', 20010, '2025-04-15', 'Battery terminal corrosion', 210.00, 60.00, 150.00),
  ('WC0074', '2024-Lysvik', 20009, '2025-04-18', 'Starter motor gear wear', 420.00, 160.00, 260.00),
  ('WC0075', '2025-Kaelori', 20007, '2025-04-20', 'Fuel pump electrical failure', 530.00, 210.00, 320.00),
  ('WC0076', '2025-Fjellstrom', 20011, '2025-04-22', 'Tire valve stem leak', 230.00, 80.00, 150.00),
  ('WC0077', '2025-Nordixa', 20010, '2025-04-25', 'Battery overheating during charging', 295.00, 100.00, 195.00),
  ('WC0078', '2025-Volkara', 20007, '2025-04-28', 'Fuel pump relay malfunction', 570.00, 220.00, 350.00),
  ('WC0079', '2025-Fjellstrom', 20011, '2025-05-01', 'Tire bead separation', 250.00, 90.00, 160.00),
  ('WC0080', '2025-Nordixa', 20009, '2025-05-03', 'Starter motor electrical short', 390.00, 140.00, 250.00),
  ('WC0081', '2025-Volkara', 20007, '2025-05-05', 'Fuel pump impeller damage', 510.00, 200.00, 310.00),
  ('WC0082', '2024-Lysvik', 20010, '2025-05-07', 'Battery case crack', 330.00, 120.00, 210.00),
  ('WC0083', '2025-Kaelori', 20011, '2025-05-09', 'Tire puncture from road debris', 210.00, 70.00, 140.00),
  ('WC0084', '2025-Nordixa', 20009, '2025-05-11', 'Starter motor brush wear', 410.00, 160.00, 250.00),
  ('WC0085', '2025-Fjellstrom', 20007, '2025-05-13', 'Fuel pump filter clog', 590.00, 230.00, 360.00),
  ('WC0086', '2025-Volkara', 20010, '2025-05-15', 'Battery cell imbalance', 350.00, 130.00, 220.00),
  ('WC0087', '2025-Fjellstrom', 20011, '2025-05-17', 'Tire slow leak from bead', 260.00, 100.00, 160.00),
  ('WC0088', '2025-Nordixa', 20009, '2025-05-19', 'Starter motor relay stuck', 430.00, 170.00, 260.00),
  ('WC0089', '2025-Volkara', 20007, '2025-05-21', 'Fuel pump wiring corrosion', 620.00, 240.00, 380.00),
  ('WC0090', '2024-Lysvik', 20011, '2025-05-23', 'Tire pressure sensor failure', 320.00, 110.00, 210.00),
  ('WC0091', '2025-Kaelori', 20010, '2025-05-25', 'Battery vent blockage', 370.00, 140.00, 230.00),
  ('WC0092', '2025-Fjellstrom', 20009, '2025-05-27', 'Starter motor mounting bolt loose', 440.00, 170.00, 270.00),
  ('WC0093', '2025-Nordixa', 20007, '2025-05-29', 'Fuel pump fuse blown', 600.00, 240.00, 360.00),
  ('WC0094', '2025-Volkara', 20011, '2025-06-01', 'Tire sidewall cut', 290.00, 100.00, 190.00),
  ('WC0095', '2025-Fjellstrom', 20010, '2025-06-03', 'Battery cable loose', 340.00, 120.00, 220.00),
  ('WC0096', '2025-Nordixa', 20009, '2025-06-05', 'Starter motor gear misalignment', 460.00, 180.00, 280.00),
  ('WC0097', '2025-Volkara', 20007, '2025-06-07', 'Fuel pump housing crack', 630.00, 250.00, 380.00),
  ('WC0098', '2024-Lysvik', 20011, '2025-06-09', 'Tire rim damage', 310.00, 120.00, 190.00),
  ('WC0099', '2025-Kaelori', 20010, '2025-06-11', 'Battery acid leak', 390.00, 150.00, 240.00),
  ('WC0100', '2025-Fjellstrom', 20009, '2025-06-13', 'Starter motor coil burnout', 470.00, 180.00, 290.00),
  ('WC0101', '2025-Nordixa', 20007, '2025-06-15', 'Fuel pump pressure drop', 610.00, 240.00, 370.00),
  ('WC0102', '2025-Volkara', 20011, '2025-06-17', 'Tire valve core failure', 300.00, 110.00, 190.00),
  ('WC0103', '2025-Fjellstrom', 20010, '2025-06-19', 'Battery sensor malfunction', 350.00, 130.00, 220.00),
  ('WC0104', '2025-Nordixa', 20009, '2025-06-21', 'Starter motor shaft fracture', 480.00, 190.00, 290.00),
  ('WC0105', '2025-Volkara', 20007, '2025-06-23', 'Fuel pump gasket leak', 640.00, 250.00, 390.00),
  ('WC0106', '2024-Lysvik', 20011, '2025-06-25', 'Tire mounting error', 320.00, 110.00, 210.00),
  ('WC0107', '2025-Kaelori', 20010, '2025-06-27', 'Battery terminal loose', 370.00, 140.00, 230.00),
  ('WC0108', '2025-Fjellstrom', 20009, '2025-06-29', 'Starter motor spring failure', 490.00, 190.00, 300.00),
  ('WC0109', '2025-Nordixa', 20007, '2025-07-01', 'Fuel pump filter tear', 620.00, 240.00, 380.00),
  ('WC0110', '2025-Volkara', 20011, '2025-07-03', 'Tire puncture from nail', 310.00, 120.00, 190.00)
;

-- COMMAND ----------

SHOW SCHEMAS IN isupply;

-- COMMAND ----------

USE CATALOG isupply;
USE SCHEMA supply_chain_silver;

-- COMMAND ----------

SHOW TABLES IN isupply.supply_chain_silver;

-- COMMAND ----------

-- MAGIC %md 
-- MAGIC **🥈 SILVER LAYER (STANDARDIZED DATA)** <br>
-- MAGIC The Silver layer applies lightweight cleansing to raw Bronze tables.  
-- MAGIC No business logic or joins are applied here — only **string cleanup and deduplication**.
-- MAGIC
-- MAGIC **🔧 Transformations Applied** <br>
-- MAGIC | ✂️ Trim        | Removes leading/trailing spaces from text fields     | <br>
-- MAGIC | 🔠 Uppercase   | Standardizes text to UPPER CASE for consistent joins | <br>
-- MAGIC | 📑 Distinct    | Removes duplicate rows from raw ingestion            | <br>
-- MAGIC | 🏷️ Type Safety | Ensures correct data types for downstream joins    | <br>
-- MAGIC
-- MAGIC **✅Implementation** <br>
-- MAGIC ✔️ `Loop through all Bronze tables`  
-- MAGIC ✔️ `Apply a reusable cleaning function`  
-- MAGIC ✔️ `Store output as Silver tables (same table names)`   
-- MAGIC <br>
-- MAGIC <img src="/Workspace/Shared/intelligent_supply_chain/supplier_scorecard/Images/silver1.png" width="160">
-- MAGIC

-- COMMAND ----------

CREATE SCHEMA IF NOT EXISTS isupply.supply_chain_silver;

-- COMMAND ----------

-- Run this once to clear the Silver schema
DROP SCHEMA isupply.supply_chain_silver CASCADE;
CREATE SCHEMA isupply.supply_chain_silver;


-- COMMAND ----------

-- MAGIC %python
-- MAGIC from pyspark.sql import functions as F
-- MAGIC
-- MAGIC bronze_schema = "isupply.supply_chain"
-- MAGIC silver_schema = "isupply.supply_chain_silver"
-- MAGIC
-- MAGIC print("🔍 Scanning Bronze schema for tables...")
-- MAGIC bronze_tables = [t.name for t in spark.catalog.listTables(bronze_schema)]
-- MAGIC
-- MAGIC # ✅ Skip internal/system tables (beginning with "_")
-- MAGIC bronze_tables = [t for t in bronze_tables if not t.startswith("_")]
-- MAGIC
-- MAGIC print(f"\nFound {len(bronze_tables)} bronze tables:")
-- MAGIC for t in bronze_tables:
-- MAGIC     print(f"   ✅ {t}")
-- MAGIC
-- MAGIC # ------------------------------------------------------------------
-- MAGIC # Cleaning rule: Trim strings, uppercase text, lowercase emails,
-- MAGIC # remove duplicates
-- MAGIC # ------------------------------------------------------------------
-- MAGIC def clean_df(df):
-- MAGIC     for col, dtype in df.dtypes:
-- MAGIC         if dtype == "string":
-- MAGIC             if "email" in col.lower():
-- MAGIC                 df = df.withColumn(col, F.lower(F.trim(F.col(col))))
-- MAGIC             else:
-- MAGIC                 df = df.withColumn(col, F.upper(F.trim(F.col(col))))
-- MAGIC     return df.dropDuplicates()
-- MAGIC
-- MAGIC # ------------------------------------------------------------------
-- MAGIC # Process tables and write to Silver
-- MAGIC # ------------------------------------------------------------------
-- MAGIC print("\n⚙️ Starting Silver table creation...\n")
-- MAGIC
-- MAGIC for table in bronze_tables:
-- MAGIC     print(f"⚙️ Processing table: {table} ...")
-- MAGIC
-- MAGIC     df = spark.table(f"{bronze_schema}.{table}")
-- MAGIC     df_clean = clean_df(df)
-- MAGIC
-- MAGIC     df_clean.write.mode("overwrite").saveAsTable(f"{silver_schema}.{table}")
-- MAGIC
-- MAGIC     print(f"   ✅ Saved Silver table: {silver_schema}.{table}  (rows={df_clean.count()})")
-- MAGIC
-- MAGIC print("\n🎉 Silver Layer Build Complete!")
-- MAGIC

-- COMMAND ----------

-- MAGIC %python
-- MAGIC from pyspark.sql import functions as F
-- MAGIC
-- MAGIC bronze_schema = "isupply.supply_chain"
-- MAGIC silver_schema = "isupply.supply_chain_silver"
-- MAGIC
-- MAGIC # Get list of tables from Bronze layer (skip system tables)
-- MAGIC bronze_tables = [t.name for t in spark.catalog.listTables(bronze_schema) if not t.name.startswith("_")]
-- MAGIC
-- MAGIC # Collect count comparisons
-- MAGIC results = []
-- MAGIC for table in bronze_tables:
-- MAGIC     bronze_count = spark.table(f"{bronze_schema}.{table}").count()
-- MAGIC     silver_count = spark.table(f"{silver_schema}.{table}").count()
-- MAGIC
-- MAGIC     status = "✅ MATCH" if bronze_count == silver_count else "❌ MISMATCH"
-- MAGIC     results.append((table, bronze_count, silver_count, status))
-- MAGIC
-- MAGIC # Convert results to DataFrame and display
-- MAGIC df_results = spark.createDataFrame(results, ["table_name", "bronze_count", "silver_count", "status"])
-- MAGIC display(df_results.orderBy("table_name"))
-- MAGIC

-- COMMAND ----------

-- MAGIC %md
-- MAGIC ⚠️ **Row Count Mismatch — supplier_sourced**
-- MAGIC
-- MAGIC <img src="/Workspace/Shared/intelligent_supply_chain/supplier_scorecard/Images/mismatch.jpg" width="100"> <img src="/Workspace/Shared/intelligent_supply_chain/supplier_scorecard/Images/surprised.avif" width="100"> 
-- MAGIC
-- MAGIC **🧾 Issue Summary**
-- MAGIC The **`supplier_sourced`** table showed a row count mismatch between the **Bronze** and **Silver** layers.
-- MAGIC
-- MAGIC **🔍 Root Cause**<br>
-- MAGIC After investigation, the mismatch is due to **duplicate records** in the Bronze layer.  
-- MAGIC The Silver transformation applies a **deduplication step**, keeping **only unique rows** across the key attributes.
-- MAGIC
-- MAGIC **This ensures:**
-- MAGIC - No duplicate sourcing transactions are carried into analytics.
-- MAGIC - Aggregations and KPIs remain accurate.
-- MAGIC
-- MAGIC **🧠 Verification Steps in below queries**

-- COMMAND ----------


SELECT *
FROM isupply.supply_chain.supplier_sourced AS b
LEFT JOIN isupply.supply_chain_silver.supplier_sourced AS s
ON b.supplier_sourced_id = s.supplier_sourced_id
    AND b.part_number = s.part_number
    AND b.supplier_number = s.supplier_number
WHERE s.supplier_sourced_id IS NULL;


-- COMMAND ----------

SELECT 
    supplier_sourced_id,
    part_number,
    supplier_number,
    COUNT(*) AS row_count
FROM isupply.supply_chain.supplier_sourced
GROUP BY supplier_sourced_id, part_number, supplier_number
HAVING COUNT(*) > 1;


-- COMMAND ----------

SELECT *
FROM isupply.supply_chain.supplier_sourced
EXCEPT
SELECT *
FROM isupply.supply_chain_silver.supplier_sourced;


-- COMMAND ----------

-- How many *extra* duplicate rows exist in Bronze?
WITH d AS (
  SELECT COUNT(*) AS cnt
  FROM isupply.supply_chain.supplier_sourced
  GROUP BY
    supplier_sourced_id, part_number, supplier_number,
    quantity_sourced, date_requested, date_received, date_shipped,
    expected_date_received, expected_date_shipped, part_price, freight_cost
  HAVING COUNT(*) > 1
)
SELECT SUM(cnt - 1) AS extra_duplicate_rows_in_bronze
FROM d;


-- COMMAND ----------

WITH numbered AS (
  SELECT
    *,
    ROW_NUMBER() OVER (
      PARTITION BY
        supplier_sourced_id, part_number, supplier_number,
        quantity_sourced, date_requested, date_received, date_shipped,
        expected_date_received, expected_date_shipped, part_price, freight_cost
      ORDER BY supplier_sourced_id
    ) AS rn
  FROM isupply.supply_chain.supplier_sourced
)
SELECT *
FROM numbered
WHERE rn > 1
ORDER BY supplier_sourced_id, part_number, supplier_number
LIMIT 100;


-- COMMAND ----------

-- Bronze distinct rows not in Silver (should be 0)
SELECT DISTINCT *
FROM isupply.supply_chain.supplier_sourced
EXCEPT
SELECT DISTINCT *
FROM isupply.supply_chain_silver.supplier_sourced;

-- Silver distinct rows not in Bronze (should be 0)
SELECT DISTINCT *
FROM isupply.supply_chain_silver.supplier_sourced
EXCEPT
SELECT DISTINCT *
FROM isupply.supply_chain.supplier_sourced;


-- COMMAND ----------

-- MAGIC %md
-- MAGIC **🥇 GOLD LAYER (Analytics | KPI Metrics)**
-- MAGIC
-- MAGIC The Gold layer contains **business-ready, aggregated datasets** used directly by dashboards and reporting tools.  
-- MAGIC Data in this layer is **fully cleaned, joined, enriched, and KPI-computed**.
-- MAGIC
-- MAGIC 📊 **Gold Tables**
-- MAGIC - ⭐ **`supplier_scorecard`** — Final consolidated table with one record per supplier, containing all KPI metrics and attributes.  
-- MAGIC - ⭐ **`supplier_scorecard_history`** — Historical snapshot table capturing supplier KPI trends across months for dashboard visualizations 
-- MAGIC                                          and trend analysis.
-- MAGIC
-- MAGIC
-- MAGIC 📈 **Key KPI Columns in `supplier_scorecard`**
-- MAGIC - `on_time_delivery_kpi` — % of orders delivered within agreed delivery window  
-- MAGIC - `on_time_shipment_kpi` — % of shipments dispatched as per plan  
-- MAGIC - `lead_time_adherence_kpi` — % of orders meeting lead-time expectations  
-- MAGIC - `defect_rate_kpi` — % of defective or returned items reported  
-- MAGIC - `supplier_risk_score` *(new)* — Weighted score based on KPI performance, representing overall supplier risk (lower is better)
-- MAGIC
-- MAGIC ✔ Final curated tables for BI tools (Power BI / Tableau / Databricks Dashboards)
-- MAGIC
-- MAGIC <br>
-- MAGIC
-- MAGIC <img src="/Workspace/Shared/intelligent_supply_chain/supplier_scorecard/Images/gold1.png" width="180">
-- MAGIC

-- COMMAND ----------

USE CATALOG isupply;
USE SCHEMA supply_chain_gold;



-- COMMAND ----------

-- ✅ Rebuild Gold Supplier Scorecard (includes lat/long)
CREATE OR REPLACE TABLE isupply.supply_chain_gold.supplier_scorecard AS
WITH src AS (
    SELECT
        supplier_number,
        quantity_sourced,
        date_requested,
        date_received,
        date_shipped,
        expected_date_received,
        expected_date_shipped
    FROM isupply.supply_chain_silver.supplier_sourced
),

totals AS (
    SELECT
        supplier_number,
        COUNT(*) AS total_deliveries,
        SUM(quantity_sourced) AS total_parts
    FROM src
    GROUP BY supplier_number
),

otd AS (
    SELECT
        supplier_number,
        COUNT(*) AS on_time_deliveries
    FROM src
    WHERE date_received <= expected_date_received
    GROUP BY supplier_number
),

ots AS (
    SELECT
        supplier_number,
        COUNT(*) AS on_time_shipments
    FROM src
    WHERE date_shipped <= expected_date_shipped
    GROUP BY supplier_number
),

lead AS (
    SELECT
        supplier_number,
        COUNT(*) AS lead_time_met
    FROM src
    WHERE DATEDIFF(date_received, date_requested) 
          <= DATEDIFF(expected_date_received, date_requested)
    GROUP BY supplier_number
),

defects AS (
    SELECT
        scnv.supplier_number,
        COUNT(*) AS defective_parts
    FROM isupply.supply_chain_silver.warranty_claims wc
    JOIN isupply.supply_chain_silver.supply_chain_network_visibility scnv
        ON wc.part_number = scnv.part_number
    GROUP BY scnv.supplier_number
)

SELECT
    sm.supplier_number,
    sm.supplier_name,
    sm.supplier_country,
    sm.latitude,
    sm.longitude,

    ROUND(100 * try_divide(otd.on_time_deliveries, totals.total_deliveries), 2) AS on_time_delivery_kpi,
    ROUND(100 * try_divide(ots.on_time_shipments, totals.total_deliveries), 2) AS on_time_shipment_kpi,
    ROUND(100 * try_divide(lead.lead_time_met, totals.total_deliveries), 2) AS lead_time_adherence_kpi,
    ROUND(100 * try_divide(defects.defective_parts, totals.total_parts), 4) AS defect_rate_kpi

FROM isupply.supply_chain_silver.supplier_master sm
LEFT JOIN totals   ON sm.supplier_number = totals.supplier_number
LEFT JOIN otd      ON sm.supplier_number = otd.supplier_number
LEFT JOIN ots      ON sm.supplier_number = ots.supplier_number
LEFT JOIN lead     ON sm.supplier_number = lead.supplier_number
LEFT JOIN defects  ON sm.supplier_number = defects.supplier_number;


-- COMMAND ----------

SELECT * FROM isupply.supply_chain_gold.supplier_scorecard;


-- COMMAND ----------

SHOW TABLES IN isupply.supply_chain_gold;

-- COMMAND ----------

SELECT COUNT(*) FROM isupply.supply_chain_gold.supplier_scorecard;

-- COMMAND ----------

SELECT * 
FROM isupply.supply_chain_gold.supplier_scorecard
ORDER BY supplier_number
LIMIT 10;

-- COMMAND ----------

SELECT COUNT(*) FROM isupply.supply_chain_gold.supplier_scorecard_history;

-- COMMAND ----------

DROP TABLE IF EXISTS isupply.supply_chain_gold.supplier_scorecard_history;


-- COMMAND ----------

-- MAGIC %python
-- MAGIC from pyspark.sql.functions import col, explode, sequence, expr, rand, round as rnd, lit
-- MAGIC import datetime
-- MAGIC
-- MAGIC gold_table = "isupply.supply_chain_gold.supplier_scorecard"
-- MAGIC history_table = "isupply.supply_chain_gold.supplier_scorecard_history"
-- MAGIC
-- MAGIC print("📌 Loading current Gold supplier_scorecard...")
-- MAGIC df_gold = spark.table(gold_table)
-- MAGIC
-- MAGIC # -------------------------------
-- MAGIC # 1️⃣ Generate Month Sequence
-- MAGIC # -------------------------------
-- MAGIC start_date = datetime.date.today().replace(day=1) - datetime.timedelta(days=59 * 30)
-- MAGIC end_date = datetime.date.today().replace(day=1)
-- MAGIC
-- MAGIC df_dates = (
-- MAGIC     spark.sql(f"""
-- MAGIC         SELECT sequence(to_date('{start_date}'), to_date('{end_date}'), interval 1 month) AS date_seq
-- MAGIC     """)
-- MAGIC     .withColumn("date", explode(col("date_seq")))
-- MAGIC     .select("date")
-- MAGIC )
-- MAGIC
-- MAGIC # -------------------------------
-- MAGIC # 2️⃣ Cross Join to Create Time-Series
-- MAGIC # -------------------------------
-- MAGIC df_ts = df_gold.crossJoin(df_dates)
-- MAGIC
-- MAGIC # -------------------------------
-- MAGIC # 2.5️⃣ Assign Downward Trend to Every Other Supplier
-- MAGIC # -------------------------------
-- MAGIC df_ts = df_ts.withColumn(
-- MAGIC     "trend_factor",
-- MAGIC     expr("CASE WHEN supplier_number % 2 = 0 THEN 0.74 ELSE 1.00 END")
-- MAGIC )
-- MAGIC
-- MAGIC # -------------------------------
-- MAGIC # 3️⃣ KPI Drift Helpers (Now Including Trend)
-- MAGIC # -------------------------------
-- MAGIC def vary(colname, pct=0.40, cap=100):
-- MAGIC     """Random ± drift + deterministic downward trend for half of suppliers"""
-- MAGIC     return rnd(
-- MAGIC         expr(
-- MAGIC             f"LEAST({cap}, {colname} * trend_factor * (1 + (rand() - 0.5) * {pct}))"
-- MAGIC         ),
-- MAGIC         2
-- MAGIC     )
-- MAGIC
-- MAGIC def vary_defect(colname, pct=0.25):
-- MAGIC     """Random drift for defects + downward trend for half of suppliers"""
-- MAGIC     return rnd(
-- MAGIC         expr(
-- MAGIC             f"GREATEST(0, LEAST(10, {colname} * trend_factor * (1 + (rand() - 0.5) * {pct})))"
-- MAGIC         ),
-- MAGIC         4
-- MAGIC     )
-- MAGIC
-- MAGIC # -------------------------------
-- MAGIC # 4️⃣ Apply KPI Drift Per Month
-- MAGIC # -------------------------------
-- MAGIC df_final = (
-- MAGIC     df_ts
-- MAGIC     .withColumn("on_time_delivery_kpi", vary("on_time_delivery_kpi"))
-- MAGIC     .withColumn("on_time_shipment_kpi", vary("on_time_shipment_kpi"))
-- MAGIC     .withColumn("lead_time_adherence_kpi", vary("lead_time_adherence_kpi"))
-- MAGIC     .withColumn("defect_rate_kpi", vary_defect("defect_rate_kpi"))
-- MAGIC     .orderBy("supplier_number", "date")
-- MAGIC )
-- MAGIC
-- MAGIC # -------------------------------
-- MAGIC # 5️⃣ Write Back to Delta Table
-- MAGIC # -------------------------------
-- MAGIC print(f"📝 Writing history data to: {history_table} ...")
-- MAGIC df_final.write.mode("overwrite").saveAsTable(history_table)
-- MAGIC
-- MAGIC print("✅ supplier_scorecard_history successfully rebuilt!")
-- MAGIC

-- COMMAND ----------

SELECT supplier_number, COUNT(*) AS months_count
FROM isupply.supply_chain_gold.supplier_scorecard_history
GROUP BY supplier_number
ORDER BY months_count DESC;


-- COMMAND ----------

USE CATALOG isupply;
USE SCHEMA supply_chain_gold;

-- COMMAND ----------

DESCRIBE TABLE isupply.supply_chain_gold.supplier_scorecard;


-- COMMAND ----------

ALTER TABLE isupply.supply_chain_gold.supplier_scorecard
ADD COLUMN supplier_risk_score DOUBLE;


-- COMMAND ----------

-- MAGIC %md
-- MAGIC **⚙️ Components of the Risk Score**<br>
-- MAGIC The **Supplier Risk Score** combines four key performance metrics using weighted contributions.<br>  
-- MAGIC A higher score indicates **greater supply risk**, while a lower score reflects **more consistent and reliable performance**.
-- MAGIC
-- MAGIC <img src="/Workspace/Shared/intelligent_supply_chain/supplier_scorecard/Images/reliability.webp" width="200" style="margin-right: 30px;"> <img src="/Workspace/Shared/intelligent_supply_chain/supplier_scorecard/Images/risk.webp" width="250"><img src="/Workspace/Shared/intelligent_supply_chain/supplier_scorecard/Images/options.jpg" width="169" style="margin-left: 50px;">
-- MAGIC
-- MAGIC ---
-- MAGIC
-- MAGIC **💡 Weight Distribution**
-- MAGIC | KPI Metric | Description | Weight |
-- MAGIC |-------------|-------------|---------|
-- MAGIC | **On-Time Delivery** | Measures delivery punctuality against expected dates | 35% |
-- MAGIC | **On-Time Shipment** | Evaluates how often shipments leave on schedule | 25% |
-- MAGIC | **Lead Time Adherence** | Tracks accuracy in meeting expected production or shipping lead time | 25% |
-- MAGIC | **Defect Rate** | Reflects quality performance based on warranty claims | 15% |
-- MAGIC
-- MAGIC ---
-- MAGIC
-- MAGIC 📘 **Interpretation:**
-- MAGIC Each metric contributes proportionally to the overall risk score.  
-- MAGIC - **Lower scores** (closer to 0) → Highly reliable suppliers with consistent delivery and quality.
-- MAGIC - **Higher scores** → Suppliers at risk of delays, production disruptions, or quality issues.
-- MAGIC

-- COMMAND ----------

UPDATE isupply.supply_chain_gold.supplier_scorecard
SET supplier_risk_score = ROUND(
        (100 - on_time_delivery_kpi) * 0.35 +
        (100 - on_time_shipment_kpi) * 0.25 +
        (100 - lead_time_adherence_kpi) * 0.25 +
        (COALESCE(defect_rate_kpi, 0) * 10) * 0.15
    , 2);


-- COMMAND ----------

SELECT supplier_number, supplier_name, supplier_risk_score
FROM isupply.supply_chain_gold.supplier_scorecard
ORDER BY supplier_risk_score DESC
LIMIT 10;


-- COMMAND ----------

-- MAGIC %md
-- MAGIC
-- MAGIC © (2025) Databricks, Inc. Use of this code is subject to your [Databricks Agreement](www.databricks.com/legal/mcsa). Use at your own risk. No warranties. No liability beyond $1,000 (as outlined at www.databricks.com/legal/db-license).